<?php
/**
 * GBX CF7 reCAPTCHA v2 Implementation
 * 
 * Replaces CF7's default reCAPTCHA v3 with v2 Checkbox logic.
 * Incorporates logic from wpcf7-recaptcha including Flamingo integration,
 * Tag Generator, and comprehensive validation.
 */

if (!defined('ABSPATH')) {
    exit;
}

class GBX_CF7_Recaptcha_V2 {

    public function __construct() {
        // Core Hooks
        add_action('setup_theme', array($this, 'gbx_setup_hooks'));
        add_action('wpcf7_init', array($this, 'gbx_add_tag_handler'), 20);
        add_action('wpcf7_admin_init', array($this, 'gbx_add_tag_generator'), 45);
        
        // Enqueues & CSS
        add_action('wp_enqueue_scripts', array($this, 'gbx_enqueue_scripts'), 50);
        add_action('wp_enqueue_scripts', array($this, 'gbx_inline_css'));
        
        // Validation & Processing
        add_filter('wpcf7_validate_recaptcha', array($this, 'gbx_validate'), 10, 2);
        add_filter('wpcf7_validate_recaptcha*', array($this, 'gbx_validate'), 10, 2);
        add_filter('wpcf7_form_tag', array($this, 'gbx_fix_tag_name'));
        add_filter('wpcf7_messages', array($this, 'gbx_custom_messages'));
        
        // Flamingo Integration
        add_action('wpcf7_submit', array($this, 'gbx_flamingo_check'), 9, 2);
    }

    /**
     * Remove v3 hooks and add v2 hooks
     */
    public function gbx_setup_hooks() {
        // Remove default v3 verification
        remove_filter('wpcf7_spam', 'wpcf7_recaptcha_verify_response', 10);
        remove_filter('wpcf7_spam', 'wpcf7_recaptcha_verify_response', 9); // Cover bases
        
        // Add v2 verification
        add_filter('wpcf7_spam', array($this, 'gbx_check_spam_score'), 9);

        // Remove default v3 enqueues
        remove_action('wp_enqueue_scripts', 'wpcf7_recaptcha_enqueue_scripts', 10);
        remove_action('wp_enqueue_scripts', 'wpcf7_recaptcha_enqueue_scripts', 20);
        
        // Remove footer JS
        remove_action('wp_footer', 'wpcf7_recaptcha_onload_script', 40);
    }

    /**
     * Replace [recaptcha] tag handler
     */
    public function gbx_add_tag_handler() {
        if (function_exists('wpcf7_remove_form_tag')) {
            wpcf7_remove_form_tag('recaptcha');
            wpcf7_add_form_tag(
                'recaptcha',
                array($this, 'gbx_render_tag'),
                array('display-block' => true)
            );
        }
    }

    /**
     * Enqueue Scripts
     */
    public function gbx_enqueue_scripts() {
        if (wp_script_is('google-recaptcha', 'registered')) {
            // Already registered? We might need to deregester if it's the v3 one?
            // Usually CF7 registers 'google-recaptcha' for v3.
            // We should overwrite it.
            wp_deregister_script('google-recaptcha');
        }

        $url = 'https://www.google.com/recaptcha/api.js';
        $url = add_query_arg(array(
            'hl'     => esc_attr(apply_filters('wpcf7_recaptcha_locale', get_locale())),
            'onload' => 'recaptchaCallback',
            'render' => 'explicit',
        ), $url);

        wp_register_script('gbx-recaptcha-controls', GBX_BLOG_SUITE_URL . 'assets/gbx-recaptcha.js', array(), '1.0', true);
        wp_register_script('google-recaptcha', $url, array('gbx-recaptcha-controls'), '2.0', true);
        
        // Enqueue global script for Block themes as per original logic
        if (function_exists('wp_is_block_theme') && wp_is_block_theme()) {
            wp_enqueue_script('google-recaptcha');
        }
    }

    /**
     * Render tag HTML
     */
    public function gbx_render_tag($tag) {
        if (!wp_script_is('google-recaptcha', 'enqueued')) {
            wp_enqueue_script('google-recaptcha');
        }

        $atts = array();
        
        // Keys
        $sitekey = $this->gbx_get_sitekey();
        
        if (empty($sitekey)) {
            return '<!-- reCAPTCHA v2 warning: No Site Key Found -->';
        }

        // Attributes
        $atts['data-sitekey'] = $sitekey;
        $atts['data-type'] = $tag->get_option('type', '(audio|image)', true);
        $atts['data-size'] = $tag->get_option('size', '(compact|normal|invisible)', true);
        $atts['data-theme'] = $tag->get_option('theme', '(dark|light)', true);
        $atts['data-align'] = $tag->get_option('align', '(left|center|right)', true);
        
        // Classes
        $class = wpcf7_form_controls_class($tag->type, 'g-recaptcha');
        $atts['class'] = $tag->get_class_option($class);
        $atts['id'] = $tag->get_id_option();

        $html = sprintf('<span %1$s></span>', wpcf7_format_atts($atts));
        $html .= $this->gbx_noscript($sitekey);
        
        return sprintf('<span class="wpcf7-form-control-wrap recaptcha" data-name="recaptcha">%s</span>', $html);
    }

    /**
     * Noscript Fallback
     */
    private function gbx_noscript($sitekey) {
        if (empty($sitekey)) return '';
        $url = 'https://www.google.com/recaptcha/api/fallback?k=' . esc_attr($sitekey);
        ob_start();
        ?>
        <noscript>
            <div class="grecaptcha-noscript">
                <iframe src="<?php echo esc_url($url); ?>" frameborder="0" scrolling="no" width="310" height="430"></iframe>
                <textarea name="g-recaptcha-response" rows="3" cols="40" placeholder="reCaptcha Response Here"></textarea>
            </div>
        </noscript>
        <?php
        return ob_get_clean();
    }

    /**
     * Spam Check (Replaces v3 verify)
     */
    public function gbx_check_spam_score($spam) {
        if ($spam) return $spam;

        $contact_form = wpcf7_get_current_contact_form();
        if (!$contact_form) return $spam;

        $tags = $contact_form->scan_form_tags(array('type' => 'recaptcha'));
        if (empty($tags)) return $spam;

        // Verify keys exist
        if (!$this->gbx_keys_exist()) return $spam;

        $token = isset($_POST['g-recaptcha-response']) ? $_POST['g-recaptcha-response'] : '';
        
        return !$this->gbx_verify_token($token);
    }
    
    /**
     * Validation Filter
     */
    public function gbx_validate($result, $tag) {
        if (empty($tag->name)) $tag->name = 'recaptcha';
        
        if (empty($_POST['g-recaptcha-response'])) {
            $msg = wpcf7_get_message('gbx_recaptcha_response_empty');
            $result->invalidate($tag, $msg ? $msg : 'Please verify that you are not a robot.');
        }
        
        return $result;
    }

    /**
     * Verify Token with Google
     */
    private function gbx_verify_token($token) {
        if (empty($token)) return false;

        $sitekey = $this->gbx_get_sitekey();
        $secret = $this->gbx_get_secret();
        
        if (empty($secret)) return false;

        $response = wp_safe_remote_post('https://www.google.com/recaptcha/api/siteverify', array(
            'body' => array(
                'secret' => $secret,
                'response' => $token,
                'remoteip' => $_SERVER['REMOTE_ADDR']
            )
        ));

        if (is_wp_error($response) || 200 != wp_remote_retrieve_response_code($response)) {
            return false;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        return isset($body['success']) && $body['success'];
    }
    
    /**
     * Flamingo Integration: Prevent Spam Logging
     */
    public function gbx_flamingo_check($contact_form, $result) {
        $recaptcha = $contact_form->scan_form_tags(array('type' => 'recaptcha'));
        
        if (!empty($recaptcha)) {
            if (empty($_POST['g-recaptcha-response'])) {
                // If reCAPTCHA is missing, unhook Flamingo so it doesn't log "Spam"
                remove_action('wpcf7_submit', 'wpcf7_flamingo_submit', 10);
            }
        }
    }

    /**
     * Admin Tag Generator
     */
    public function gbx_add_tag_generator() {
        if (class_exists('WPCF7_TagGenerator')) {
            $tag_generator = WPCF7_TagGenerator::get_instance();
            $tag_generator->add(
                'recaptcha',
                'reCaptcha (v2)',
                array($this, 'gbx_tag_generator_ui'),
                array('version' => 2)
            );
        }
    }

    public function gbx_tag_generator_ui($contact_form, $options = array()) {
        $options = wp_parse_args($options, array());
        $tgg = new WPCF7_TagGeneratorGenerator($options['content']);
        ?>
        <header class="description-box">
            <h3>reCaptcha v2 Tag Generator</h3>
            <p>Generates a "I'm not a robot" checkbox widget.</p>
        </header>

        <div class="control-box">
            <?php 
            $tgg->print('field_type', array(
                'with_required' => false,
                'select_options' => array($options['id'] => 'ReCaptcha v2')
            )); 
            ?>
            <fieldset>
                <legend>Size</legend>
                <label><input type="radio" name="size" class="option default" value="normal" checked="checked"> Normal</label><br>
                <label><input type="radio" name="size" class="option" value="compact"> Compact</label>
            </fieldset>

            <fieldset>
                <legend>Theme</legend>
                <label><input type="radio" name="theme" class="option default" value="light" checked="checked"> Light</label><br>
                <label><input type="radio" name="theme" class="option" value="dark"> Dark</label>
            </fieldset>
            
            <?php $tgg->print('id_attr'); ?>
            <?php $tgg->print('class_attr'); ?>
        </div>

        <footer class="insert-box">
            <?php $tgg->print('insert_box_content'); ?>
        </footer>
        <?php
    }

    // --- Utilities ---

    public function gbx_inline_css() {
        wp_add_inline_style('contact-form-7', '.wpcf7 .wpcf7-recaptcha iframe {margin-bottom: 0;} .wpcf7 .wpcf7-recaptcha[data-align="center"] > div {margin: 0 auto;}');
    }
    
    public function gbx_fix_tag_name($tag) {
        if (empty($tag['name']) && 'recaptcha' === $tag['type']) {
            $tag['name'] = 'recaptcha';
        }
        return $tag;
    }
    
    public function gbx_custom_messages($messages) {
        return array_merge($messages, array(
            'gbx_recaptcha_response_empty' => array(
                'description' => 'Message when reCaptcha is unchecked.',
                'default' => 'Please verify that you are not a robot.'
            )
        ));
    }
    
    // --- Key Helpers ---
    
    private function gbx_keys_exist() {
        return $this->gbx_get_sitekey() && $this->gbx_get_secret();
    }

    private function gbx_get_sitekey() {
        if (defined('WPCF7_RECAPTCHA_SITEKEY')) return WPCF7_RECAPTCHA_SITEKEY;
        $service = WPCF7::get_option('recaptcha');
        return (!empty($service) && is_array($service)) ? array_keys($service)[0] : false;
    }

    private function gbx_get_secret() {
        if (defined('WPCF7_RECAPTCHA_SECRET')) return WPCF7_RECAPTCHA_SECRET;
        $service = WPCF7::get_option('recaptcha');
        if (!empty($service) && is_array($service)) {
            $key = array_keys($service)[0];
            return $service[$key];
        }
        return false;
    }
}

// Initialize
new GBX_CF7_Recaptcha_V2();
