<?php
/**
 * Internal Links Class
 * Handles smart internal linking functionality
 */

if (!defined('ABSPATH')) {
    exit;
}

class GBX_Internal_Links {
    
    public function __construct() {
        add_action('add_meta_boxes', array($this, 'add_meta_box'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_sil_analyze', array($this, 'ajax_analyze'));
        add_action('wp_ajax_sil_add_link', array($this, 'ajax_add_link'));
    }
    
    public function add_meta_box() {
        $screen = get_current_screen();
        if ($screen && $screen->post_type === 'post') {
            add_meta_box('smart_internal_links_box', 'Smart Internal Links', array($this, 'render_meta_box'), 'post', 'normal', 'low');
        }
    }
    
    public function render_meta_box($post) {
        wp_nonce_field('sil_meta_box', 'sil_nonce_field');
        ?>
        <div id="sil-container" class="sil-metabox">
            <div class="sil-table-wrapper">
                <table id="sil-table" class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th class="column-anchor">Anchor Text</th>
                            <th class="column-strength">Match Strength</th>
                            <th class="column-target">Target Post</th>
                            <th class="column-action">Action</th>
                        </tr>
                    </thead>
                    <tbody id="sil-tbody"></tbody>
                </table>
            </div>
            
            <div class="sil-actions">
                <button type="button" id="sil-analyze" class="button button-primary">
                    <span class="dashicons dashicons-search"></span> Analyze Content
                </button>
            </div>
        </div>
        <input type="hidden" id="sil-post-id" value="<?php echo $post->ID; ?>" />
        <?php
    }
    
    public function enqueue_scripts($hook) {
        if ($hook !== 'post.php' && $hook !== 'post-new.php') {
            return;
        }
        
        wp_enqueue_style('sil-style', GBX_BLOG_SUITE_URL . 'assets/sil-style.css', array(), GBX_BLOG_SUITE_VERSION);
        wp_enqueue_script('sil-script', GBX_BLOG_SUITE_URL . 'assets/sil-script.js', array('jquery'), GBX_BLOG_SUITE_VERSION, true);
        wp_localize_script('sil-script', 'silData', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('sil_nonce')
        ));
    }
    
    public function ajax_analyze() {
        check_ajax_referer('sil_nonce', 'nonce');
        
        $post_id = intval($_POST['post_id']);
        $post = get_post($post_id);
        
        if (!$post || $post->post_type !== 'post') {
            wp_send_json_error('Invalid post');
        }
        
        $content = $this->normalize($post->post_content);
        $phrases_3 = $this->get_phrases($content, 3);
        $phrases_2 = $this->get_phrases($content, 2);
        
        $posts = get_posts(array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'numberposts' => -1,
            'exclude' => array($post_id)
        ));
        
        $suggestions = array();
        
        // 3-word matches first
        foreach ($phrases_3 as $phrase) {
            if (count($suggestions) >= 5) break;
            
            $is_linked = $this->is_already_linked($post->post_content, $phrase);
            
            foreach ($posts as $p) {
                $title = $this->normalize($p->post_title);
                if (strpos($title, $phrase) !== false) {
                    $suggestions[] = array(
                        'anchor' => $phrase,
                        'strength' => 'Strong',
                        'target_id' => $p->ID,
                        'target_title' => $p->post_title,
                        'target_url' => get_permalink($p->ID),
                        'is_linked' => $is_linked
                    );
                    break;
                }
            }
        }
        
        // 2-word matches if needed
        if (count($suggestions) < 5) {
            foreach ($phrases_2 as $phrase) {
                if (count($suggestions) >= 5) break;
                
                $is_linked = $this->is_already_linked($post->post_content, $phrase);
                
                foreach ($posts as $p) {
                    $title = $this->normalize($p->post_title);
                    if (strpos($title, $phrase) !== false) {
                        $suggestions[] = array(
                            'anchor' => $phrase,
                            'strength' => 'Normal',
                            'target_id' => $p->ID,
                            'target_title' => $p->post_title,
                            'target_url' => get_permalink($p->ID),
                            'is_linked' => $is_linked
                        );
                        break;
                    }
                }
            }
        }
        
        wp_send_json_success($suggestions);
    }
    
    public function ajax_add_link() {
        check_ajax_referer('sil_nonce', 'nonce');
        
        $post_id = intval($_POST['post_id']);
        $anchor = sanitize_text_field($_POST['anchor']);
        $target_url = esc_url($_POST['target_url']);
        
        $post = get_post($post_id);
        if (!$post) wp_send_json_error('Invalid post');
        
        $content = $post->post_content;
        $updated = $this->insert_link($content, $anchor, $target_url);
        
        if ($updated === false) {
            wp_send_json_error('Could not insert link');
        }
        
        wp_update_post(array(
            'ID' => $post_id,
            'post_content' => $updated
        ));
        
        wp_send_json_success(array('content' => $updated));
    }
    
    private function normalize($text) {
        $text = wp_strip_all_tags($text);
        $text = mb_strtolower($text, 'UTF-8');
        $text = preg_replace('/[^\p{L}\p{N}\s]/u', ' ', $text);
        $text = preg_replace('/\s+/', ' ', $text);
        return trim($text);
    }
    
    private function get_phrases($text, $word_count) {
        $words = explode(' ', $text);
        $phrases = array();
        
        for ($i = 0; $i <= count($words) - $word_count; $i++) {
            $phrase = implode(' ', array_slice($words, $i, $word_count));
            if (strlen($phrase) > 3) {
                $phrases[] = $phrase;
            }
        }
        
        return array_unique($phrases);
    }
    
    private function is_already_linked($content, $anchor) {
        $pattern = '/<a[^>]*>.*?' . preg_quote($anchor, '/') . '.*?<\/a>/i';
        return preg_match($pattern, $content) > 0;
    }
    
    private function insert_link($content, $anchor, $url) {
        $pos = stripos($content, $anchor);
        if ($pos === false) return false;
        
        $actual_anchor = substr($content, $pos, strlen($anchor));
        $link = '<a href="' . esc_url($url) . '"><strong>' . $actual_anchor . '</strong></a>';
        return substr_replace($content, $link, $pos, strlen($anchor));
    }
}
